/***************************************************************************/
/**                                                                       **/
/**                               MAGICARD                                **/
/**                                                                       **/
/***************************************************************************/
/**                                                                       **/
/**  PROJECT      : Magicard SDK                                          **/
/**                                                                       **/
/**  MODULE NAME  : MagAPI.h                                              **/
/**                                                                       **/
/**  SIN No.      : 2118                                                  **/
/**                                                                       **/
/**  COPYRIGHT    : Magicard                                              **/
/**                                                                       **/
/***************************************************************************/

#ifndef __MAGAPI_H__
#define __MAGAPI_H__

#if _MSC_VER > 1000
  #pragma once
#endif

#ifdef __cplusplus
extern "C"
{
#endif // __cplusplus

#ifdef _COMPILING_MAGSDK
  #define LIBSPEC __declspec(dllexport)
#else
  #define LIBSPEC __declspec(dllimport)
#endif // _COMPILING_MAGSDK

//#############################################################################
//##### GENERAL STRUCTURES AND ENUM DEFINITIONS
//#############################################################################

//Stop these negative enum values generating erroneous compiler warnings
#pragma warning (push)
#pragma warning (disable:4341)

typedef enum tag_ID_RETURN:signed char
{
    ID_SUCCESS              = ERROR_SUCCESS, // Defined in terms of the Win API value
    ID_TIMEOUT              = -1,            // Timed out waiting for a response from the printer
    ID_ERROR                = -2,            // Non specific SDK error
    ID_PRINTER_ERROR        = -3,            // Printer has reported an error
    ID_DRIVER_NOTCOMPLIANT  = -4,            // Driver is not compliant with the SDK DLL
    ID_OPENPRINTER_ERROR    = -5,            // Error opening communication with the printer
    ID_REMOTECOMM_ERROR     = -6,            // Remote communications error
    ID_LOCALCOMM_ERROR      = -7,            // Local communications error
    ID_SPOOLER_NOT_EMPTY    = -8,            // Printer is already printing a job
    ID_SESSION_ACTIVE       = -9,            // Session already active
    ID_LOCALCOMM_IN_USE     = -10,           // Local communications error
    ID_PARAM_ERROR          = -11,           // Input parameter out of range
    ID_INVALID_SESSION      = -12,           // Invalid SDK session handle
    ID_MORE_DATA            = -13,           // Buffer is too small
	ID_INVALID_PRINTER      = -14,			 // Invalid Action for this type of printer
	ID_INVALID_ACTION       = -15,			 // Printer is not equipped for this action
	ID_READ_ERROR           = -16,			 // Cannot read this parameter in this format
	ID_WRITE_ERROR          = -17,			 // Cannot write this parameter
	ID_NO_MAG_DATA          = -18,			 // No data from magnetic stripe
	ID_NO_APDU_DATA			= -19,			 // No APDU data was returned
} ID_RETURN;

#pragma warning (pop)           // restore the warning

typedef enum tag_ID_CARDSIDE:BYTE
{
    ID_SIDE_FRONT,
    ID_SIDE_BACK,
} ID_SIDE;

typedef enum tag_ID_READWRITE:BYTE
{
    ID_READ,
    ID_WRITE
} ID_READWRITE;

// Custom driver escape to indicate SDK support
#define ESC_IS_SDK_CAPABLE  9001

#define ESC_GETDEVMODE      15000
#define ESC_GETDEVMODESIZE  15002

//Basic Colour definitions
#define ID_COLOUR_RED     RGB(255,0,0)
#define ID_COLOUR_GREEN   RGB(0,255,0)
#define ID_COLOUR_BLUE    RGB(0,0,255)
#define ID_COLOUR_CYAN    RGB(0,255,255)
#define ID_COLOUR_MAGENTA RGB(255,0,255)
#define ID_COLOUR_YELLOW  RGB(255,255,0)
#define ID_COLOUR_WHITE   RGB(255,255,255)
#define ID_COLOUR_BLACK   RGB(0,0,0)

#define ID_TRANSPARENT    HOLLOW_BRUSH

//#############################################################################
//##### FUNCTION DEFINITIONS
//#############################################################################

/******************************************************************************
 * Initializes the SDK and its communication channel
 ******************************************************************************/
typedef enum tag_ID_CONFIG:BYTE
{
    ID_CONFIG_NORMAL,      // Normal operation in the case of errors
    ID_CONFIG_QUIET        // Do not show status monitor for errors
} ID_CONFIG;

LIBSPEC ID_RETURN WINAPI ID_OpenSession
(
    HDC         hDC,       // Device Context handle for the printer driver
    HANDLE     *phSession, // Pointer to a handle that will identify the new Session
    ID_CONFIG   dwFlags
);

typedef ID_RETURN (WINAPI * PFNIDOPEN)(HDC, HANDLE *, ID_CONFIG);

/******************************************************************************
 * Closes the communications channel with the status monitor, returns the status
 * monitor to its normal behaviour and releases all resources used
 ******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_CloseSession(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDCLOSE)(HANDLE);

/******************************************************************************
 * Instructs the printer to feed a card to the specified chip encoding position
 ******************************************************************************/
typedef enum tag_ID_FEED:BYTE
{
    ID_FEED_CHIPCARD,
    ID_FEED_CONTACTLESS
} ID_FEED;

LIBSPEC ID_RETURN WINAPI ID_FeedCard
(
    HANDLE  hSession, // Handle to the active Session
    ID_FEED dwMode,   // The card type
    BYTE    iParam    // An integer parameter that modifies the feed card
                      // position (Range 0-99)
);

typedef ID_RETURN (WINAPI * PFNIDFEED)(HANDLE, ID_FEED, BYTE);

/******************************************************************************
 * Instructs the printer to eject any card that may be present in the mechanism
 ******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_EjectCard(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDEJECT)(HANDLE);

/******************************************************************************
 * Waits until the printer reports that it is no longer busy or until a time-out
 * period elapses.
 * This function should be expected to time out during lengthy operations. It is
 * up to the application to determine how long it is going to wait, by repeating
 * calls to this function, before deciding that the printer is not responding.
 * The timeout is set at 30 seconds.
 ******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_WaitForPrinter(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDWAIT)(HANDLE);

LIBSPEC int WINAPI ID_PrepareForPrint(HANDLE hSession);
typedef ID_RETURN (WINAPI * PFNIDPREPARE)(HANDLE);
LIBSPEC int WINAPI ID_WaitPrintComplete(HANDLE hSession);
typedef ID_RETURN (WINAPI * PFNIDPRINTCOMP)(HANDLE);

/******************************************************************************
 * Retrieves a string containing the last status message sent by the printer
 ******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_LastMessage
(
    HANDLE  hSession,     // Handle to the active Session
    LPTSTR  lpBuffer,     // Pointer to buffer to be filled with the status message
    LPDWORD pdwBufferSize // Pointer to a variable containing the buffer size
);

typedef ID_RETURN (WINAPI * PFNIDLASTMSG)(HANDLE, LPTSTR, LPDWORD);

#if defined(_UNICODE) || defined(UNICODE)
  #define IDFN_LASTMESSAGE "ID_LastMessageW"
  #define ID_LastMessage   ID_LastMessageW
#else
  #define IDFN_LASTMESSAGE "ID_LastMessageA"
  #define ID_LastMessage   ID_LastMessageA
#endif

/******************************************************************************
 * Send the given command string to the printer
 ******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_GeneralCommand
(
    HANDLE  hSession,     // Handle to the active Session
    LPTSTR  CommandString // The ASCII CommandString to be sent to the printer
);

typedef ID_RETURN (WINAPI * PFNIDGENERAL)(HANDLE, LPTSTR);

#if defined(_UNICODE) || defined(UNICODE)
  #define IDFN_GENCOMMAND	"ID_GeneralCommandW"
  #define ID_GeneralCommand	ID_GeneralCommandW
#else
  #define IDFN_GENCOMMAND	"ID_GeneralCommandA"
  #define ID_GeneralCommand	ID_GeneralCommandA
#endif

/******************************************************************************
 * Writes the given magnetic stripe data to the card
 ******************************************************************************/
#define NUM_MAG_TRACKS 3	//< The number of magnetic tracks

typedef enum tag_ID_ENCODING:BYTE
{
    ID_ENCODING_ISO,
    ID_ENCODING_JIS2,
    ID_ENCODING_RAW
} ID_ENCODING;

typedef enum tag_ID_VERIFY:BYTE
{
    ID_VERIFY_OFF,
    ID_VERIFY_ON
} ID_VERIFY;

typedef enum tag_ID_COERCIVITY:BYTE
{
    ID_COERCIVITY_DEFAULT,
    ID_COERCIVITY_HICO,
    ID_COERCIVITY_LOCO,
	ID_COERCIVITY_MIDCO
} ID_COERCIVITY;

typedef enum tag_ID_BITSPERCHAR:BYTE
{
    ID_BITSPERCHAR_DEFAULT,
    ID_BITSPERCHAR_1,
    ID_BITSPERCHAR_4,
    ID_BITSPERCHAR_5,
    ID_BITSPERCHAR_7
} ID_BITSPERCHAR;

typedef enum tag_ID_BITSPERINCH:BYTE
{
    ID_BITSPERINCH_DEFAULT,
    ID_BITSPERINCH_75,
    ID_BITSPERINCH_210
} ID_BITSPERINCH;

typedef enum tag_ID_PARITY:BYTE
{
    ID_PARITY_DEFAULT,
    ID_PARITY_OFF,
    ID_PARITY_ODD,
    ID_PARITY_EVEN
} ID_PARITY;

typedef enum tag_ID_LRC:BYTE
{
    ID_LRC_DEFAULT,
    ID_LRC_OFF,
    ID_LRC_ODD,
    ID_LRC_EVEN
} ID_LRC;

typedef struct tag_MagTrack
{
	int				iCharCount;
	LPTSTR          lpszData;
    ID_BITSPERCHAR  iBitsPerChar;
    ID_BITSPERINCH  iBitsPerInch;
    ID_PARITY       iParity;
    ID_LRC          iLRC;
} ID_MAGTRACK, *PID_MAGTRACK;

typedef struct tag_MagDef
{
    ID_ENCODING   iEncodingSpec;
    ID_COERCIVITY iCoercivity;
    ID_VERIFY     iVerify;
	ID_MAGTRACK	  Track[NUM_MAG_TRACKS];
} ID_MAGDEF, *PID_MAGDEF;

LIBSPEC ID_RETURN WINAPI ID_EncodeMag(HANDLE hSession, PID_MAGDEF pEncode);

typedef ID_RETURN (WINAPI * PFNIDENCODEMAG)(HANDLE hSession, PID_MAGDEF pEncode);

#if defined(_UNICODE) || defined(UNICODE)
  #define IDFN_ENCODEMAG "ID_EncodeMagW"
  #define ID_EncodeMag   ID_EncodeMagW
#else
  #define IDFN_ENCODEMAG "ID_EncodeMagA"
  #define ID_EncodeMag   ID_EncodeMagA
#endif

/******************************************************************************
 * Reads the Magnetic Stripe on the card, using the given encoding specification
 ******************************************************************************/
typedef struct tag_RAW_DATA
{
    char tk1[172];      // ISO max is 79  (7bpc, 210bpi)
    char tk2[172];      // ISO max is 40  (5bpc, 75bpi)
    char tk3[172];      // ISO max is 107 (5bpc, 210bpi)
} RAW_DATA;

typedef struct tag_ID_MAGDATA
{
    DWORD    msv_id;     // Unique ID to distinguish this message
    DWORD    msg_len;    // size of message, including this
    DWORD    tk1_pass;   // TRUE if the track passed; FALSE if failed or not tested
    DWORD    tk2_pass;
    DWORD    tk3_pass;
    DWORD    tk1_len;    // Number of bytes returned for this track...
    DWORD    tk2_len;    // ...from start sentinel to LRC inclusive
    DWORD    tk3_len;
    RAW_DATA raw;        // Raw data for each track, which has been verified
} ID_MAGDATA, *PID_MAGDATA;

LIBSPEC ID_RETURN WINAPI ID_ReadMag
(
    HANDLE       hSession,   // Handle to the active Session
    VOID	    *pData,      // Pointer to the data Structure
    ID_ENCODING  iEncodingSpec
);

typedef ID_RETURN (WINAPI * PFNIDREADMAG)(HANDLE, VOID *, ID_ENCODING);

LIBSPEC ID_RETURN WINAPI ID_ReadMagTracks
(
    HANDLE       hSession,     // Handle to the active Session
    VOID        *pData,        // Pointer to the data Structure
    ID_ENCODING  iEncodingSpec,
    BYTE         TrackMask     // Bit Mask indicating tracks to read
);

typedef ID_RETURN (WINAPI * PFNIDMAGTRACKS)(HANDLE, VOID *, ID_ENCODING, BYTE);

/******************************************************************************
* Structures for Reading Magnetic Data (Ultima & Rio Pro360 Printers)
******************************************************************************/
#define MAG_BUFSIZE      112	// The size of the track data buffer
#define DEFAULT_MAGSTART 8000

typedef struct
{
	char data[MAG_BUFSIZE];	//< The binary data to be encoded on this track.
} TRACK_DATA;

typedef struct
{
   TRACK_DATA track[NUM_MAG_TRACKS]; //< An array of mag data structures, one for each track.
} ID_MAGDATA2, *PID_MAGDATA2;

/******************************************************************************
* Function for Reading/Writing magnetic encoding start offset
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_MagStart
(
    HANDLE           hSession,
    ID_READWRITE     ReadWrite,
    DWORD           *pOffset
);
typedef ID_RETURN (WINAPI * PFNIDMAGSTART)(HANDLE, ID_READWRITE, DWORD *);

/******************************************************************************
 * Reads the current printer status
 ******************************************************************************/
typedef enum tag_ID_STATUS:BYTE
{
    ID_STATUS_READY,
    ID_STATUS_BUSY,
    ID_STATUS_ERROR,
    ID_STATUS_OFFLINE
} ID_STATUS;

LIBSPEC int WINAPI ID_PrinterStatus(HANDLE hSession);

typedef int (WINAPI * PFNIDSTATUS)(HANDLE);

/******************************************************************************
 * Reports the Current Status Information message from a printer
******************************************************************************/

#define SERIAL_SIZE     20
#define ULTIMA_STRING   32

//Masks used in iBitFields
#define ID_MASK_ENCODEDEFAULT   (1 << 0) //0x01
#define ID_MASK_ENCODEPLATEN    (1 << 1) //0x02
#define ID_MASK_ERASESPEED      (1 << 2) //0x04
#define ID_MASK_EJECTMODE       (1 << 3) //0x08
#define ID_MASK_HORZEJECT       (1 << 4) //0x10
#define ID_MASK_ENCODEXLI       (1 << 5) //0x20

#define ID_MASK_ENCODEMODE (ID_MASK_ENCODEDEFAULT | ID_MASK_ENCODEPLATEN | ID_MASK_ENCODEXLI)

typedef struct tag_PRINTER_INFO
{
    BOOL    bPrinterConnected;
    DWORD   eModel;
    char    sModel[30];
    DWORD   ePrintheadType;
    char    sPrinterSerial[SERIAL_SIZE];
    char    sPrintheadSerial[SERIAL_SIZE];
    char    sPCBSerial[SERIAL_SIZE];
    WCHAR   sFirmwareVersion[SERIAL_SIZE];

    DWORD   iDummy1;
    DWORD   iDummy2;
    DWORD   iSmartOffset;
    DWORD   iBitFields;
    DWORD   iDensity;

    DWORD   iHandFeed;
    DWORD   iCardsPrinted;
    DWORD   iCardsOnPrinthead;
    DWORD   iDyePanelsPrinted;
    DWORD   iCleansSinceShipped;
    DWORD   iDyePanelsSinceClean;
    DWORD   iCardsSinceClean;
    DWORD   iCardsBetweenCleans;

    DWORD   iPrintHeadPosn;
    DWORD   iImageStartPosn;
    DWORD   iImageEndPosn;
    DWORD   iMajorError;
    DWORD   iMinorError;

    char    sTagUID[20];
    DWORD   iShotsOnFilm;
    DWORD   iShotsUsed;
    char    sDyeFilmType[20];
    DWORD   iColourLength;
    DWORD   iResinLength;
    DWORD   iOvercoatLength;
    DWORD   eDyeFlags;
    DWORD   iCommandCode;
    DWORD   iDOB;
    DWORD   eDyeFilmManuf;
    DWORD   eDyeFilmProg;
} PRINTER_INFO, *PPRINTER_INFO;

typedef struct tag_ULTIMA_INFO
{
	char firmware_version[ULTIMA_STRING];
	char printer_serial[ULTIMA_STRING];
	char model_name[ULTIMA_STRING];
	long printer_partner_code;
	long capability_flags;
	char manufacturer[ULTIMA_STRING];

	long transfer_target_temperature;
	long transfer_actual_temperature;

	long total_cards;
	long total_panels;
	long cards_on_printhead;
	long panels_on_printhead;

	long cards_low;
	char colour_name[ULTIMA_STRING];
	long colour_total_prints;
	long colour_prints_remaining;
	long colour_partner_code;
	char transfer_name[ULTIMA_STRING];
	long transfer_total_prints;
	long transfer_prints_remaining;
	long transfer_partner_code;

	long error_code;
	long error_context;

	long cards_between_prompts;
	long cleaning_overdue;
	long cards_since_clean;
	long panels_since_clean;
	long cleaning_cycles;

	long laminate_actual_temperature;
	long laminate_target_temperature;
	long laminate_type;
	long card_speed;

	char reserved[32];
} ULTIMA_INFO, *PULTIMA_INFO;

LIBSPEC ID_RETURN WINAPI ID_PrinterInfo
(
    HANDLE   hSession,    // Handle to the active Session
    VOID	*pPrinterInfo // Pointer to the Printer Info Structure
);

typedef ID_RETURN (WINAPI * PFNIDPRINTERINFO)(HANDLE, VOID *);

/******************************************************************************
 * Instructs the printer to print an internal test pattern
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_PrintTestCard(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDPRINTTEST) (HANDLE);

/******************************************************************************
 * Instructs the printer to execute a cleaning cycle.
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_CleanPrinter(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDCLEAN)(HANDLE);

/******************************************************************************
 * Instructs the printer to perform a reset cycle.
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_RestartPrinter(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDRESTART)(HANDLE);

/******************************************************************************
 * Instructs the printer to rotate the card using the flipper mechanism.
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_FlipCard(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDFLIP)(HANDLE);

/******************************************************************************
 * Instructs the printer to perform an erase cycle on a rewritable card.
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_EraseCard
(
    HANDLE  hSession,     // Handle to the active Session
	DWORD   Count,        // Number of cards to be erased
    DWORD   iBottomLeftX, // Coordinates for the area to be erased
    DWORD   iBottomLeftY,
    DWORD   iTopRightX,
    DWORD   iTopRightY
);

typedef ID_RETURN (WINAPI * PFNIDERASE)(HANDLE, DWORD, DWORD, DWORD, DWORD, DWORD);

/******************************************************************************
 * Returns the card dimensions for the printer
******************************************************************************/
LIBSPEC int WINAPI ID_CardWidth(HANDLE hSession);
typedef int (WINAPI * PFNIDCARDWIDTH)(HANDLE);

LIBSPEC int WINAPI ID_CardHeight(HANDLE hSession);
typedef int (WINAPI * PFNIDCARDHEIGHT)(HANDLE);

/******************************************************************************
 * Returns the bit count of the  SDK DLL.
******************************************************************************/
LIBSPEC BYTE WINAPI ID_SDKBits();

typedef ID_RETURN (WINAPI * PFNIDSDKBITS)();

/******************************************************************************
 * Returns the version of the SDK DLL.
******************************************************************************/
typedef struct tag_SDKVERSION
{
    DWORD Major;
    DWORD Minor;
    DWORD Build;
    DWORD Private;
} SDKVERSION, *PSDKVERSION;

LIBSPEC ID_RETURN WINAPI ID_SDKVersion
(
    HANDLE      hSession, // Handle to the active Session
    PSDKVERSION pVersion  // Pointer to the SDK Version structure
);

typedef ID_RETURN (WINAPI * PFNIDVERSION)(HANDLE, PSDKVERSION);

/******************************************************************************
 * Sends a response to an error condition to the printer
******************************************************************************/
typedef enum tag_ID_RESPONSE:BYTE
{
    ID_RESPONSE_OK,
    ID_RESPONSE_CAN
} ID_RESPONSE;

LIBSPEC ID_RETURN WINAPI ID_Response
(
    HANDLE      hSession, // Handle to the active Session
    ID_RESPONSE iResponse
);

typedef ID_RETURN (WINAPI * PFNIDRESPONSE)(HANDLE, ID_RESPONSE);

/******************************************************************************
 * Returns an identifier of the printer family
******************************************************************************/
typedef enum tag_ID_PRINTER:BYTE
{
    ID_RIOTANGO,
    ID_AOTA,
    ID_ENDURO,          //Enduro Family (incl. Rio Pro)
	ID_ULTIMA,
	ID_GENERATION2,
} ID_PRINTER;

LIBSPEC ID_PRINTER WINAPI ID_PrinterType(HANDLE hSession);

typedef DWORD (WINAPI * PFNIDPRNTYPE)(HANDLE);

/******************************************************************************
 * Returns an identifier of the printer model & capability
******************************************************************************/
typedef enum tag_ID_MODEL:DWORD
{
	ID_FN_MAGNETIC    = 0x01,
	ID_FN_DUPLEX      = 0x02,
	ID_FN_REWRITE     = 0x04,
	ID_FN_CONTACTLESS = 0x08,
	ID_FN_CHIP        = 0x10,
	ID_MOD_PRONTO     = 0x20,
	//                  0x40
	//                  0x80
	ID_FN_XTENDED     = 0x100,
	ID_MOD_ENDURO     = 0x200,
	ID_MOD_PRONTO100  = 0x400,
	ID_FN_ETHERNET    = 0x800,
	ID_MOD_ULTIMA     = 0x1000,
	ID_FN_LAMINATE    = 0x2000,
	ID_FN_600DPI      = 0x4000,
	ID_MOD_RIOPRO     = 0x8000,
	ID_MOD_PRO360     = 0x10000,
	ID_MOD_600        = 0x20000,
	ID_MOD_300        = 0x40000,
	ID_MOD_GEN2       = 0x80000,
} ID_MODEL;

LIBSPEC DWORD WINAPI ID_PrinterModel(HANDLE hSession);

typedef DWORD (WINAPI * PFNIDPRNMODEL)(HANDLE);

/******************************************************************************
 * Returns an identifier of the type of connection to the printer
******************************************************************************/
typedef enum tag_ID_PORT:BYTE
{
    ID_PORT_UNKNOWN,
    ID_PORT_USB,
    ID_PORT_RESERVED,
    ID_PORT_TCPIP,
    ID_PORT_FILE
} ID_PORT;

LIBSPEC ID_PORT WINAPI ID_ConnectionType(HANDLE hSession);

typedef ID_PORT (WINAPI * PFNIDCONNECTION)(HANDLE);

//#############################################################################

/******************************************************************************
 * Function for Reading/Writing Print Settings
******************************************************************************/
typedef enum tag_ID_PRINTSIDES:BYTE
{
    ID_FRONT_ONLY,
    ID_BOTH_SIDES,
    ID_BACK_ONLY
} ID_PRINTSIDES;

typedef struct tag_PRINTSET
{
    ID_PRINTSIDES Duplex;
    BYTE          CopyCount;
    DWORD         CardSize;
} PRINTSET, *PPRINTSET;

LIBSPEC ID_RETURN WINAPI ID_PrintSettings
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PPRINTSET    pPrintSettings
);

typedef ID_RETURN (WINAPI * PFNIDPRINTSET)(HANDLE, ID_READWRITE, PPRINTSET);

/******************************************************************************
* Function for Reading/Writing Colour Format
******************************************************************************/
typedef enum tag_ID_COLOURFORMAT:BYTE
{
    ID_COLOURFORMAT_YMC,
    ID_COLOURFORMAT_YMCK,
    ID_COLOURFORMAT_K
} ID_COLOURFORMAT;

typedef enum tag_ID_ORIENTATION:BYTE
{
    ID_ORIENTATION_PORTRAIT,
    ID_ORIENTATION_LANDSCAPE
} ID_ORIENTATION;

typedef struct tag_CARDSET
{
    ID_COLOURFORMAT ColourFormat;
    BOOL            Overcoat;
    ID_ORIENTATION  Orientation;
    BOOL            Rotation;
} CARDSET, *PCARDSET;

LIBSPEC ID_RETURN WINAPI ID_CardSettings
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    ID_SIDE      SideID,
    PCARDSET     pCardParams
);

typedef ID_RETURN (WINAPI * PFNIDCARDSET)(HANDLE, ID_READWRITE, ID_SIDE, PCARDSET);

/******************************************************************************
* Function for Reading/Writing HoloKote Settings
******************************************************************************/
typedef enum tag_ID_ROTATION:BYTE
{
    ID_ROTATION_NONE,
    ID_ROTATION_90,
    ID_ROTATION_180,
    ID_ROTATION_270
} ID_ROTATION;

typedef struct tag_HOLOKOTE
{
    BYTE        ImageID;
    DWORD       Map;
    ID_ROTATION Rotation;
    BOOL        DisableCustomKey;
    BOOL        UseLaminate;
} HOLOKOTE, *PHOLOKOTE;

LIBSPEC ID_RETURN WINAPI ID_HoloKote
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    ID_SIDE      SideID,
    PHOLOKOTE    pHolokote
);

typedef ID_RETURN (WINAPI * PFNIDHOLOKOTE)(HANDLE, ID_READWRITE, ID_SIDE, PHOLOKOTE);

/******************************************************************************
* Function for Reading/Writing HoloPatch Settings
******************************************************************************/
typedef struct tag_HOLOPATCH
{
    BYTE   Position;
    BOOL   ColourHole;
} HOLOPATCH, *PHOLOPATCH;

LIBSPEC ID_RETURN WINAPI ID_HoloPatch
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PHOLOPATCH   pHolopatch
);

typedef ID_RETURN (WINAPI * PFNIDHOLOPATCH)(HANDLE, ID_READWRITE, PHOLOPATCH);

/******************************************************************************
* Function for Reading/Writing Areas and Holes
******************************************************************************/
#define MAX_AREAHOLENO 2

typedef enum tag_ID_AREAHOLE:BYTE
{
    ID_AREA,
    ID_HOLE
} ID_AREAHOLE;

typedef struct tag_AREA
{
    DWORD Left;       //Area/Hole Left X co-ordinate
    DWORD Width;      //  :   :   Width
    DWORD Bottom;     //  :   :   Bottom Y co-ordinate
    DWORD Height;     //  :   :   Height
} AREA, *PAREA;

LIBSPEC ID_RETURN WINAPI ID_AreaHole
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    ID_SIDE      SideID,
    ID_AREAHOLE  AreaHoleType,
    BYTE         AreaHoleID,
    PAREA        pAreaHole
);

typedef ID_RETURN (WINAPI * PFNIDAREAHOLE)(HANDLE,
                                           ID_READWRITE,
										   ID_SIDE,
										   ID_AREAHOLE,
										   BYTE,
										   PAREA);

/******************************************************************************
* Function for Reading/Writing Resin Options
******************************************************************************/
typedef struct tag_K_OPTIONS
{
	BOOL PicturesUseYMC;
	BOOL BlackText;
	BOOL MonoBitmaps;
	BOOL BlackPolygons;
	BOOL AllBlackIsYMC;
} K_OPTIONS, *PK_OPTIONS;

LIBSPEC ID_RETURN WINAPI ID_ResinOptions
(
	HANDLE       hSession,
	ID_READWRITE ReadWrite,
	ID_SIDE      SideID,
	PK_OPTIONS   pOptions
);

typedef ID_RETURN(WINAPI * PFNIDRESINOPTIONS)(HANDLE, ID_READWRITE, ID_SIDE, PK_OPTIONS);

/******************************************************************************
* Function for Reading/Writing Resin Areas
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_ResinArea
(
	HANDLE       hSession,
	ID_READWRITE ReadWrite,
	ID_SIDE      SideID,
	BYTE         AreaID,
	PAREA        pArea
);

typedef ID_RETURN(WINAPI * PFNIDRESINAREA)(HANDLE, ID_READWRITE, ID_SIDE, BYTE, PAREA);

/******************************************************************************
* Function for Reading/Writing Colour Correction
******************************************************************************/
typedef enum tag_ID_COLCORR:BYTE
{
    ID_NONE,
    ID_GAMMA1,
    ID_GAMMA2,
    ID_ICC_INTERNAL,
    ID_ICC_EXTERNAL,
    ID_PERCEPTUAL,
    ID_SATURATION,
    ID_RELCOLORIMETRIC,
    ID_ABSCOLORIMETRIC
} COLCORRECT, *PCOLCORRECT;

LIBSPEC ID_RETURN WINAPI ID_ColourCorrection
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PCOLCORRECT  pCorrection
);

typedef ID_RETURN (WINAPI * PFNIDCORRECTION)(HANDLE, ID_READWRITE, PCOLCORRECT);

/******************************************************************************
* Function for Reading/Writing Colour Areas
******************************************************************************/

LIBSPEC ID_RETURN WINAPI ID_ColourArea
(
	HANDLE       hSession,
    ID_READWRITE ReadWrite,
    ID_SIDE      SideID,
    BYTE         AreaID,
    PAREA        pArea,
    PCOLCORRECT  pCorrection
);

typedef ID_RETURN (WINAPI * PFNIDCOLOURAREA)(HANDLE,
											 ID_READWRITE,
											 ID_SIDE,
											 BYTE,
											 PAREA,
											 PCOLCORRECT);

/******************************************************************************
* Function for Reading/Writing Sharpness
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_Sharpness
(
    HANDLE        hSession,
    ID_READWRITE  ReadWrite,
    signed char  *pSharpness
);

typedef ID_RETURN (WINAPI * PFNIDSHARPNESS)(HANDLE, ID_READWRITE, signed char *);

/******************************************************************************
* Function for Reading/Writing Print Speed
******************************************************************************/
typedef enum tag_ID_PRINTSPEED:BYTE
{
    ID_HIGHSPEED,
    ID_NORMALSPEED
} ID_PRINTSPEED, *PID_PRINTSPEED;

LIBSPEC ID_RETURN WINAPI ID_PrintSpeed
(
    HANDLE         hSession,
    ID_READWRITE   ReadWrite,
    PID_PRINTSPEED pPrintSpeed
);

typedef ID_RETURN (WINAPI * PFNIDPRINTSPEED)(HANDLE, ID_READWRITE, PID_PRINTSPEED);

/******************************************************************************
* Function for Reading/Writing Power Levels
******************************************************************************/
typedef struct tag_POWERLEVEL
{
    BYTE YMC;
    BYTE Resin;
    BYTE Overcoat;
} POWERLEVEL, *PPOWERLEVEL;

LIBSPEC ID_RETURN WINAPI ID_PowerLevel
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PPOWERLEVEL  pPowerLevel
);

typedef ID_RETURN (WINAPI * PFNIDPOWERLEVEL)(HANDLE, ID_READWRITE, PPOWERLEVEL);

/******************************************************************************
* Function for Reading/Writing Rewritable Card Settings
******************************************************************************/
typedef struct tag_REWRITABLE
{
    BOOL    EraseBeforePrint;
    AREA    EraseArea;
    BYTE    ErasePowerStart;
    BYTE    ErasePowerEnd;
    BYTE    WritePower;
} REWRITABLE, *PREWRITABLE;

LIBSPEC ID_RETURN WINAPI ID_Rewritable
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PREWRITABLE  pRewritable
);

typedef ID_RETURN (WINAPI * PFNIDREWRITABLE)(HANDLE, ID_READWRITE, PREWRITABLE);

/******************************************************************************
* Function for Defining Rewritable Areas
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_RewritableArea
(
	HANDLE       hSession,
	ID_READWRITE ReadWrite,
	ID_SIDE      SideID,
	BYTE         AreaID,
	PAREA        pArea
);

typedef ID_RETURN(WINAPI * PFNIDREWRITEAREA)(HANDLE, ID_READWRITE, ID_SIDE, BYTE, PAREA);

/******************************************************************************
* Function for Setting/Providing Password for restricted access operation
******************************************************************************/
typedef enum tag_ID_PASSWORD:BYTE
{
    ID_PWDSET,
    ID_PWDUSE
} ID_PASSWORDCMD;

LIBSPEC ID_RETURN WINAPI ID_Password
(
    HANDLE         hSession,
    ID_PASSWORDCMD Command,
    LPTSTR         Password1,
    LPTSTR         Password2
);

typedef ID_RETURN (WINAPI * PFNIDPASSWORD)(HANDLE, ID_PASSWORDCMD, LPTSTR, LPTSTR);

#if defined(_UNICODE) || defined(UNICODE)
  #define IDFN_PASSWORD "ID_PasswordW"
  #define ID_Password   ID_PasswordW
#else
  #define IDFN_PASSWORD "ID_PasswordA"
  #define ID_Password   ID_PasswordA
#endif

/******************************************************************************
* Function for Reading/Writing Ethernet Settings
******************************************************************************/
typedef enum tag_ID_ADDRESSING:BYTE
{
    ID_STATIC,
    ID_DYNAMIC
} ID_ADDRESSING;

typedef struct tag_IPDATA
{
    ID_ADDRESSING IPAddressMode;
    int           IPAddress;
    int           SubnetMask;
    int           Gateway;
	int			  MACAddrHi;
	int			  MACAddrLo;
} IPDATA, *PIPDATA;

LIBSPEC ID_RETURN WINAPI ID_IPSettings
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PIPDATA      pIPData
);

typedef ID_RETURN (WINAPI * PFNIDIPSETTINGS)(HANDLE, ID_READWRITE, PIPDATA);

/******************************************************************************
* Function for Reading/Writing Control of the driver GUI
******************************************************************************/
typedef struct tag_ID_GUICONTROL
{
    BOOL GUIUser;
    BOOL GUIPrinter;
} ID_GUICONTROL, *PID_GUICONTROL;

LIBSPEC ID_RETURN WINAPI ID_GUIControl
(
    HANDLE         hSession,
    ID_READWRITE   ReadWrite,
    PID_GUICONTROL pGUIControl
);

typedef ID_RETURN (WINAPI * PFNIDGUICONTROL)(HANDLE, ID_READWRITE, PID_GUICONTROL);

/******************************************************************************
* Function for Reading/Writing assorted control bits/flags
******************************************************************************/
typedef enum tag_ID_CONTROL:DWORD
{
    ID_ENCODE_ONLY = 0x01,
    ID_ISO7810     = 0x02,
	ID_LOGGING     = 0x04
} ID_CONTROL, *PID_CONTROL;

LIBSPEC ID_RETURN WINAPI ID_Control
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PID_CONTROL  pControl
);
typedef ID_RETURN (WINAPI * PFNIDCONTROL)(HANDLE, ID_READWRITE, PID_CONTROL);

/******************************************************************************
* Function for Reading/Writing the printing resolution
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_Resolution
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    LPBYTE	     pResolution
);
typedef ID_RETURN (WINAPI * PFNIDRESOLUTION)(HANDLE, ID_READWRITE, LPBYTE);

/******************************************************************************
* Function for Reading/Writing Hand Feed Setting
******************************************************************************/
typedef enum tag_ID_HANDFEED:BYTE
{
    ID_HANDFEED_OFF,
    ID_HANDFEED_ON
} ID_HANDFEED, *PID_HANDFEED;

LIBSPEC ID_RETURN WINAPI ID_HandFeed
(
    HANDLE       hSession,
    ID_READWRITE ReadWrite,
    PID_HANDFEED pHandFeed
);

typedef ID_RETURN (WINAPI * PFNIDHANDFEED)(HANDLE, ID_READWRITE, PID_HANDFEED);

/******************************************************************************
 * Function for Reading/Writing Eject Mode
 ******************************************************************************/
typedef enum tag_ID_EJECTMODE:BYTE
{
    ID_EJECT_OFF,
    ID_EJECT_ON
} ID_EJECTMODE, *PID_EJECTMODE;

LIBSPEC ID_RETURN WINAPI ID_EjectMode
(
    HANDLE        hSession,
    ID_READWRITE  ReadWrite,
    PID_EJECTMODE iMode
);

typedef ID_RETURN (WINAPI * PFNIDEJECTMODE)(HANDLE, ID_READWRITE, PID_EJECTMODE);

/******************************************************************************
* Function for Reading/Writing Horizontal Eject
******************************************************************************/
typedef enum tag_ID_HORZEJECT:BYTE
{
    ID_HORZEJECT_OFF,
    ID_HORZEJECT_ON
} ID_HORZEJECT, *PID_HORZEJECT;

LIBSPEC ID_RETURN WINAPI ID_HorzEject
(
    HANDLE        hSession,
    ID_READWRITE  ReadWrite,
    PID_HORZEJECT HorzEjectMode
);

typedef ID_RETURN (WINAPI * PFNIDHORZEJECT)(HANDLE, ID_READWRITE, PID_HORZEJECT);

/******************************************************************************
* Function for Reading/Writing Smart Encoding Mode
******************************************************************************/
typedef enum tag_ID_SMARTMODE:BYTE
{
    ID_SMART_DEFAULT,
    ID_SMART_PLATEN,
	ID_SMART_XLI
} ID_SMARTMODE, *PID_SMARTMODE;

LIBSPEC ID_RETURN WINAPI ID_SmartMode
(
    HANDLE        hSession,
    ID_READWRITE  ReadWrite,
    PID_SMARTMODE SmartMode
);

typedef ID_RETURN (WINAPI * PFNIDSMARTMODE)(HANDLE, ID_READWRITE, PID_SMARTMODE);

/******************************************************************************
* Function for Reading/Writing Smart Encoding Offset position
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_SmartOffset
(
    HANDLE        hSession,
    ID_READWRITE  ReadWrite,
    BYTE         *SmartOffset
);

typedef ID_RETURN (WINAPI * PFNIDSMARTOFFSET)(HANDLE, ID_READWRITE, BYTE *);

/******************************************************************************
* Function for Reading/Writing Erase Speed for Rewritable cards
******************************************************************************/
typedef enum tag_ID_ERASESPEED:BYTE
{
    ID_ERASE_THOROUGH,
    ID_ERASE_QUICK
} ID_ERASESPEED, *PID_ERASESPEED;

LIBSPEC ID_RETURN WINAPI ID_EraseSpeed
(
    HANDLE         hSession,
    ID_READWRITE   ReadWrite,
    PID_ERASESPEED EraseSpeed
);

typedef ID_RETURN (WINAPI * PFNIDERASESPEED)(HANDLE, ID_READWRITE, PID_ERASESPEED);

/******************************************************************************
* Function for Reading/Writing the Colour Adjustment Settings of the driver
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_ColourAdjust
(
    HANDLE           hSession,
    ID_READWRITE     ReadWrite,
    PCOLORADJUSTMENT pColourAdj
);

typedef ID_RETURN (WINAPI * PFNIDCOLADJ)(HANDLE, ID_READWRITE, PCOLORADJUSTMENT);

//#############################################################################
//##### SDK PRINTING FUNCTIONS
//#############################################################################

/******************************************************************************
* Initialises the specified canvas, returning the HDC to the calling application
******************************************************************************/
typedef enum tag_ID_CANVAS:BYTE
{
    ID_CANVAS_FRONT,
    ID_CANVAS_BACK,
    ID_CANVAS_FRONT_RESIN,
    ID_CANVAS_BACK_RESIN
} ID_CANVAS;

LIBSPEC ID_RETURN WINAPI ID_CanvasInit
(
    HANDLE     hSession,
    HDC       *pHDC,
    ID_CANVAS  CanvasID
);

typedef ID_RETURN (WINAPI * PFNIDINIT)(HANDLE, HDC *, ID_CANVAS);

/******************************************************************************
* Deletes the specified canvas
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_CanvasDelete
(
    HANDLE    hSession,
    ID_CANVAS CanvasID
);

typedef ID_RETURN (WINAPI * PFNIDDELETE)(HANDLE, ID_CANVAS);

/******************************************************************************
* Draws text on the specified canvas
******************************************************************************/

#define ID_TEXT_BOLD      0x01
#define ID_TEXT_ITALIC    0x02
#define ID_TEXT_UNDERLINE 0x04
#define ID_TEXT_STRIKEOUT 0x08

typedef struct tag_ID_TEXTDEF
{
    TCHAR *Text;
    DWORD  X;
    DWORD  Y;
    DWORD  Angle;
    TCHAR *FontName;
    DWORD  Size;
    DWORD  Colour;
    DWORD  Style;
} ID_TEXTDEF, *PID_TEXTDEF;

LIBSPEC ID_RETURN WINAPI ID_DrawTextA(HANDLE hSession, ID_CANVAS Canvas, PID_TEXTDEF TextDef);
LIBSPEC ID_RETURN WINAPI ID_DrawTextW(HANDLE hSession, ID_CANVAS Canvas, PID_TEXTDEF TextDef);

typedef ID_RETURN (WINAPI * PFNIDTEXT)(HANDLE, ID_CANVAS, PID_TEXTDEF);

#if defined(_UNICODE) || defined(UNICODE)
    #define IDFN_DRAWTEXT    "ID_DrawTextW"
#else
    #define IDFN_DRAWTEXT    "ID_DrawTextA"
#endif

/******************************************************************************
* Draws tilde encoded data for magnetic encoding on the canvas
******************************************************************************/

typedef struct tag_ID_MAGTEXT
{
	int    Track;
	TCHAR *Data;
} ID_MAGTEXT, *PID_MAGTEXT;

LIBSPEC ID_RETURN WINAPI ID_DrawMagTextA(HANDLE hSession, ID_SIDE Side, PID_MAGTEXT MagDef);
LIBSPEC ID_RETURN WINAPI ID_DrawMagTextW(HANDLE hSession, ID_SIDE Side, PID_MAGTEXT MagDef);

typedef ID_RETURN (WINAPI * PFNIDMAGTEXT)(HANDLE, ID_SIDE, PID_MAGTEXT);

#if defined(_UNICODE) || defined(UNICODE)
    #define IDFN_MAGTEXT "ID_DrawMagTextW"
#else
    #define IDFN_MAGTEXT "ID_DrawMagTextA"
#endif

/******************************************************************************
* Draws a shape on the specified canvas
******************************************************************************/
typedef enum tag_ID_SHAPES:BYTE
{
    ID_SHAPE_RECTANGLE,
    ID_SHAPE_ELLIPSE,
    ID_SHAPE_ROUNDRECT,
    ID_SHAPE_PIE,
    ID_SHAPE_CHORD
} ID_SHAPE;

typedef struct tag_ID_SHAPE
{
    ID_SHAPE Shape;
    DWORD    PenColour;
    DWORD    PenWidth;
    DWORD    FillColour;
    RECT     Bound;
    DWORD    p1;
    DWORD    p2;
    DWORD    p3;
    DWORD    p4;
} ID_SHAPEDEF, *PID_SHAPEDEF;

LIBSPEC ID_RETURN WINAPI ID_DrawShape(HANDLE hSession, ID_CANVAS CanvasID, PID_SHAPEDEF pIDShape);

typedef ID_RETURN (WINAPI * PFNIDSHAPE)(HANDLE, ID_CANVAS, PID_SHAPEDEF);

/******************************************************************************
* Draws a line on the specified canvas
******************************************************************************/
typedef struct tag_ID_LINE
{
    DWORD   Colour;
    DWORD   Width;
    DWORD   StartX;
    DWORD   StartY;
    DWORD   EndX;
    DWORD   EndY;
} ID_LINEDEF, *PID_LINEDEF;

LIBSPEC ID_RETURN WINAPI ID_DrawLine(HANDLE hSession, ID_CANVAS CanvasID, PID_LINEDEF LineDef);

typedef ID_RETURN (WINAPI * PFNIDLINE)(HANDLE, ID_CANVAS, PID_LINEDEF);

/******************************************************************************
* Draws an image (BMP, JPG, PNG, TIF) on the specified canvas
******************************************************************************/
typedef struct tag_ID_IMAGEDEF
{
    TCHAR *Filename;
    DWORD  X;
    DWORD  Y;
    DWORD  p1;
    DWORD  p2;
} ID_IMAGEDEF, *PID_IMAGEDEF;

LIBSPEC ID_RETURN WINAPI ID_DrawImageA(HANDLE hSession, ID_CANVAS CanvasID, PID_IMAGEDEF ImageDef);
LIBSPEC ID_RETURN WINAPI ID_DrawImageW(HANDLE hSession, ID_CANVAS CanvasID, PID_IMAGEDEF ImageDef);

typedef ID_RETURN (WINAPI * PFNIDIMAGE)(HANDLE, ID_CANVAS, PID_IMAGEDEF);

#if defined(_UNICODE) || defined(UNICODE)
  #define IDFN_DRAWIMAGE    "ID_DrawImageW"
#else
  #define IDFN_DRAWIMAGE    "ID_DrawImageA"
#endif

/******************************************************************************
* Prints the card using the canvas(ses) currently set up
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_PrintCard(HANDLE hSession);

typedef ID_RETURN (WINAPI * PFNIDPRINT)(HANDLE);

/******************************************************************************
* Gets the current devmode
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_GetDevmode(HANDLE hSession,	LPDEVMODE pBuffer, int *size);

typedef ID_RETURN (WINAPI * PFNIDGETDEV)(HANDLE, LPDEVMODE, int *);

/******************************************************************************
* Sets the current devmode
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_SetDevmode(HANDLE hSession,	LPDEVMODE pBuffer, int size);

typedef ID_RETURN (WINAPI * PFNIDSETDEV)(HANDLE, LPDEVMODE, int);

/******************************************************************************
* Updates the current device context
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_UpdateDC(HANDLE hSession, HDC hDC);

typedef ID_RETURN (WINAPI * PFNIDUPDC)(HANDLE, HDC);

/******************************************************************************
* Displays the Printer Preferences dialog, and updates driver settings
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_PrinterPrefs(HANDLE hSession, HWND hWnd);

typedef ID_RETURN (WINAPI * PFNIDPPREFS)(HANDLE, HWND);

//##################################################################################################################################

//############################
//##### ULTIMA FUNCTIONS #####
//############################

typedef enum tag_ID_CARD_POSITION
{
	ID_TRANSFER_STANDBY,    // 0 - move the card to the transfer standby position
	ID_HEATED_ROLLER,       // 1 - move the card to the heated roller position
	ID_FLIP_STANDBY,        // 2 - move the card to the flip standby position
	ID_ROTATE,              // 3 - rotate the card and return it to the transfer standby position
	ID_MAG_ENCODER,         // 4 - move the card to the mag encoder
	ID_CONTACT_ENCODER,     // 5 - move the card to the contact smart encoder
	ID_EJECT,               // 6 - eject the card
	ID_REJECT,              // 7 - reject the card
	ID_LAMINATOR,           // 8 - give the card to the laminator
	ID_CONTACTLESS_ENCODER, // 9 - give the card to the contactless smart encoder
	ID_INITIALISE           //10 - Clear the print engine of any cards
} ID_CARD_POSITION;

#define ID_STANDBY ID_TRANSFER_STANDBY

LIBSPEC ID_RETURN WINAPI ID_MoveCard(HANDLE hSession, ID_CARD_POSITION Location);
typedef ID_RETURN (WINAPI * PFNIDMOVE)(HANDLE, ID_CARD_POSITION);

/******************************************************************************
* Function for Reading HoloKote Identities
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_HoloKoteIdentityA
(
    HANDLE  hSession,
	BYTE    SlotNo,
    LPSTR   pBuffer,
	DWORD  *Size
);

LIBSPEC ID_RETURN WINAPI ID_HoloKoteIdentityW
(
    HANDLE  hSession,
	BYTE    SlotNo,
    LPWSTR  pBuffer,
	DWORD  *Size
);

typedef ID_RETURN (WINAPI * PFNIDHKIDENTITY)(HANDLE, BYTE, LPTSTR, DWORD *);

#if defined(_UNICODE) || defined(UNICODE)
  #define IDFN_HOLOKOTEIDENTITY "ID_HoloKoteIdentityW"
#else
  #define IDFN_HOLOKOTEIDENTITY "ID_HoloKoteIdentityA"
#endif

/******************************************************************************
* Function for checking whether temperature preheat is in progress
******************************************************************************/
typedef enum tag_ID_TEMPERATURE:BYTE
{
    ID_TEMPERATURE_TRANSFER,
    ID_TEMPERATURE_LAMINATE
} ID_TEMPERATURE;

LIBSPEC ID_RETURN WINAPI ID_Temperature
(
    HANDLE         hSession,
    ID_TEMPERATURE TempID,
	LPBOOL		   Status
);
typedef ID_RETURN (WINAPI * PFNIDTEMP)(HANDLE, ID_TEMPERATURE, LPBOOL);

//############################
//##### PRO360 FUNCTIONS #####
//############################

/******************************************************************************
* Function for determining the location of a card in the printer (if any)
******************************************************************************/
typedef enum tag_ID_LOCATION:BYTE
{
	ID_UNKNOWN,
	ID_NO_CARD,
	ID_PRINT_READY,
	ID_ENCODER,
} ID_LOCATION, *PID_LOCATION;

LIBSPEC ID_RETURN WINAPI ID_CardLocation
(
    HANDLE       hSession,
	PID_LOCATION location
);
typedef ID_RETURN (WINAPI * PFNIDLOCATION)(HANDLE, PID_LOCATION);

/******************************************************************************
* Function for obtaining the HoloKote preview(s)
******************************************************************************/

#define ID_HKKEYLEN         256
#define ID_HKPREVIEW_HEIGHT 166
#define ID_HKPREVIEW_WIDTH  259
//Preview size padded to 32 byte boundary
#define ID_HKPREVIEW_SIZE   ((((ID_HKPREVIEW_WIDTH * ID_HKPREVIEW_HEIGHT) + 31) & ~31) / 8)

typedef struct tag_HKPREVIEW
{
    BYTE key[ID_HKKEYLEN];
    BYTE data[ID_HKPREVIEW_SIZE];
} HKPREVIEW, *PHKPREVIEW;

LIBSPEC ID_RETURN WINAPI ID_HoloKotePreview
(
    HANDLE      hSession,
    PHKPREVIEW  pBuffer,
	LPDWORD     size
);
typedef ID_RETURN (WINAPI * PFNIDHKTPREVIEW)(HANDLE, PHKPREVIEW, LPDWORD);

/******************************************************************************
* Function for obtaining the HoloKote count
******************************************************************************/

LIBSPEC int WINAPI ID_HoloKoteCount(HANDLE hSession);
typedef int (WINAPI * PFNIDHKTCOUNT)(HANDLE);

/******************************************************************************
* Function for Defining the Printable Area
******************************************************************************/
LIBSPEC ID_RETURN WINAPI ID_PrintableArea
(
	HANDLE       hSession,
	ID_READWRITE ReadWrite,
	PAREA        pArea
);

typedef ID_RETURN(WINAPI * PFNIDPRINTAREA)(HANDLE, ID_READWRITE, PAREA);

/******************************************************************************
* Function for reading the Sensor states
******************************************************************************/

typedef enum tag_ID_SENSOR:BYTE
{
	ID_LID_SENSOR          = 0x01,
	ID_HOPPER_SENSOR       = 0x02,
	ID_HOPPER_SENSOR_VALID = 0x04,
	ID_EXIT_SENSOR         = 0x08,
	ID_EXIT_SENSOR_VALID   = 0x10,
} ID_SENSOR;

LIBSPEC ID_RETURN WINAPI ID_Sensors
(
	HANDLE       hSession,
	LPDWORD      pState
);

typedef ID_RETURN(WINAPI * PFNIDSENSORS)(HANDLE, LPDWORD);

/******************************************************************************
* Function for moving the film to a specific dye panel
******************************************************************************/

typedef enum tag_ID_PANEL:BYTE
{
	ID_PANEL_INIT,
	ID_PANEL_NEXT,
	ID_PANEL_Y,
	ID_PANEL_M,
	ID_PANEL_C,
	ID_PANEL_K,
	ID_PANEL_O
} ID_PANEL;

LIBSPEC ID_RETURN WINAPI ID_MoveFilm
(
    HANDLE   hSession,
	ID_PANEL panel
);

typedef ID_RETURN(WINAPI * PFNIDMOVEFILM)(HANDLE, ID_PANEL);

//##################################################################################################################################

typedef enum tag_PRO360_PARAM:BYTE
{
    OP_CAPABILITY_DUPLEX,
    OP_CAPABILITY_MAG,
	OP_CAPABILITY_SMART,
	OP_VERSION_SYSTEM,
	OP_VERSION_PRINTER_APP,
	OP_VERSION_FIRMWARE,
	OP_VERSION_MAG_FIRMWARE,
	OP_SERIAL_PRINTER,
	OP_SERIAL_PRINTHEAD,
	OP_SERIAL_BOARD,
	OP_MODEL_NAME,
	OP_PRINTER_DEALER_CODE,
	OP_PRINTER_PROJECT_CODE,
	OP_STATS_TOTAL_CARDS,
	OP_STATS_TOTAL_PANELS,
	OP_STATS_PRINTHEAD_CARDS,
	OP_STATS_PRINTHEAD_PANELS,
	OP_STATS_CARDS_SINCE_CLEAN,
	OP_STATS_PANELS_SINCE_CLEAN,
	OP_STATS_CLEANS,
	OP_DYE_NAME,
	OP_DYE_SERIAL,
	OP_DYE_PRINTS_TOTAL,
	OP_DYE_PRINTS_REMAINING,
	OP_DYE_MANUFACTURER,
	OP_CLEAN_CARDS_INTERVAL,
	OP_CLEAN_OVERDUE,
	OP_NET_MAC_ADDRESS,
	OP_NET_USE_DHCP,
	OP_NET_IP_ADDRESS,
	OP_NET_GATEWAY,
	OP_NET_SUBNET,
	OP_NET_STATIC_IP,
	OP_NET_STATIC_GATEWAY,
	OP_NET_STATIC_SUBNET,
	OP_ERROR,
	OP_HAND_FEED,
	OP_CARD_LOCATION,
	OP_MAG_START,
	OP_YMC_LENGTH,
	OP_K_LENGTH,
	OP_O_LENGTH,
    OP_SMART_OFFSET,
	OP_HOLOKOTE_SLOTS,
	OP_CAPABILITY_600DPI,
	OP_HOPPER_SENSOR,
	OP_EXIT_SENSOR,
	OP_LID_SENSOR,
	OP_MANUFACTURER,
	OP_HOPPER_SENSOR_FITTED,
	OP_EXIT_SENSOR_FITTED,
	OP_REGION,

	MAX_PRO360_PARAM
} PRO360_PARAM;

LIBSPEC ID_RETURN WINAPI ID_AccessInt(
	HANDLE       hSession,
    ID_READWRITE ReadWrite,
	PRO360_PARAM param_id,
	int         *ptr
);
typedef ID_RETURN (WINAPI * PFNIDACCINT)(HANDLE, ID_READWRITE, PRO360_PARAM, int *);

LIBSPEC ID_RETURN WINAPI ID_AccessBool
(
	HANDLE       hSession,
    ID_READWRITE ReadWrite,
	PRO360_PARAM param_id,
	bool        *ptr
);
typedef ID_RETURN (WINAPI * PFNIDACCBOOL)(HANDLE, ID_READWRITE, PRO360_PARAM, bool *);

LIBSPEC ID_RETURN WINAPI ID_AccessBuffer
(
	HANDLE       hSession,
    ID_READWRITE ReadWrite,
	PRO360_PARAM param_id,
	char        *ptr,
	int         *size
);
typedef ID_RETURN (WINAPI * PFNIDACCBUFF)(HANDLE, ID_READWRITE, PRO360_PARAM, char *, int *);

#ifdef _MSC_VER
LIBSPEC ID_RETURN WINAPI ID_AccessString
(
	HANDLE       hSession,
    ID_READWRITE ReadWrite,
	PRO360_PARAM param_id,
	CString&     string
);
typedef ID_RETURN (WINAPI * PFNIDACCSTRING)(HANDLE, ID_READWRITE, PRO360_PARAM, CString&);

#endif
/******************************************************************************
* Function for sending/receiving APDU Smart Card Command
******************************************************************************/

LIBSPEC ID_RETURN WINAPI ID_SendAPDU
(
	HANDLE       hSession,
	char*     inputString,
	char*	 outputString,
	int		 *size
);
typedef ID_RETURN (WINAPI * PFNIDSENDAPDU)(HANDLE, char* , char*, int *);

//##################################################################################################################################

//============================
//== DEPRECATED DEFINITIONS ==
//============================

#define ID_RIOPRO360 ID_GENERATION2

//These definitions are deprecated, but maintain backwards compatibility with API functions.
#define MAGICARD_SUCCESS             ID_SUCCESS
#define MAGICARD_TIMEOUT             ID_TIMEOUT
#define MAGICARD_ERROR               ID_ERROR
#define MAGICARD_PRINTER_ERROR       ID_PRINTER_ERROR
#define MAGICARD_DRIVER_NOTCOMPLIANT ID_DRIVER_NOTCOMPLIANT
#define MAGICARD_OPENPRINTER_ERROR   ID_OPENPRINTER_ERROR
#define MAGICARD_REMOTECOMM_ERROR    ID_REMOTECOMM_ERROR
#define MAGICARD_LOCALCOMM_ERROR     ID_LOCALCOMM_ERROR
#define MAGICARD_SPOOLER_NOT_EMPTY   ID_SPOOLER_NOT_EMPTY
#define MAGICARD_REMOTECOMM_IN_USE   ID_SESSION_ACTIVE
#define MAGICARD_LOCALCOMM_IN_USE    ID_LOCALCOMM_IN_USE

#define MSVDATA  ID_MAGDATA
#define PMSVDATA PID_MAGDATA

#define ESC_IS_API_CAPABLE ESC_IS_SDK_CAPABLE

BOOL __inline MAGICARD_Is_Status_Reporting_In_Use(int iError)
{
    return (MAGICARD_REMOTECOMM_IN_USE == iError || MAGICARD_LOCALCOMM_IN_USE == iError);
}

#define CONFIG_NORMAL ID_CONFIG_NORMAL
#define CONFIG_QUIET  ID_CONFIG_QUIET
LIBSPEC int WINAPI EnableStatusReporting(HDC, HANDLE *, DWORD);
typedef int (WINAPI * PFNENABLESTATUSREPORTING)(HDC, HANDLE*, DWORD);

LIBSPEC int WINAPI DisableStatusReporting(HANDLE);
typedef int (WINAPI * PFNDISABLESTATUSREPORTING)(HANDLE);

LIBSPEC int WINAPI RequestMagData(HANDLE);
typedef int (WINAPI * PFNREQUESTMAGDATA)(HANDLE);

LIBSPEC int WINAPI ReadMagData(HANDLE, PMSVDATA);
typedef int (WINAPI * PFNREADMAGDATA)(HANDLE, PMSVDATA);

enum {FEED_CHIPCARD = 1, FEED_CONTACTLESS}; // 0 is unused in the API
LIBSPEC int WINAPI FeedCardA(HANDLE, DWORD, int, LPSTR);
LIBSPEC int WINAPI FeedCardW(HANDLE, DWORD, int, LPWSTR);
typedef int (WINAPI * PFNFEEDCARDA)(HANDLE, DWORD, int, LPSTR);
typedef int (WINAPI * PFNFEEDCARDW)(HANDLE, DWORD, int, LPWSTR);
#if defined(_UNICODE) || defined(UNICODE)
  #define FeedCard FeedCardW
  #define PFNFEEDCARD PFNFEEDCARDW
#else
  #define FeedCard FeedCardA
  #define PFNFEEDCARD PFNFEEDCARDA
#endif

LIBSPEC int WINAPI EjectCardA(HANDLE, LPSTR);
LIBSPEC int WINAPI EjectCardW(HANDLE, LPWSTR);
typedef int (WINAPI * PFNEJECTCARDA)(HANDLE, LPSTR);
typedef int (WINAPI * PFNEJECTCARDW)(HANDLE, LPWSTR);
#if defined(_UNICODE) || defined(UNICODE)
  #define EjectCard EjectCardW
  #define PFNEJECTCARD PFNEJECTCARDW
#else
  #define EjectCard EjectCardA
  #define PFNEJECTCARD PFNEJECTCARDA
#endif

LIBSPEC int WINAPI WaitForPrinter(HANDLE);
typedef int (WINAPI * PFNWAITFORPRINTER)(HANDLE);

LIBSPEC int WINAPI GetLastPrinterMessage(HANDLE, LPSTR, LPDWORD);
typedef int (WINAPI * PFNGETLASTPRINTERMESSAGE)(HANDLE, LPSTR, LPDWORD);

LIBSPEC int WINAPI GetLastEnduroMessage(HANDLE, LPTSTR, LPDWORD);
typedef int (WINAPI * PFNGETLASTENDUROMESSAGE)(HANDLE, LPTSTR, LPDWORD);

LIBSPEC int WINAPI GeneralCommand(HANDLE, LPSTR);
typedef int (WINAPI * PFNGENERALCOMMAND)(HANDLE, LPSTR);

#define EMS_ENCODING_SPEC_ISO   ID_ENCODING_ISO
#define EMS_ENCODING_SPEC_JIS2  ID_ENCODING_JIS2
#define EMS_VERIFY_OFF          ID_VERIFY_OFF
#define EMS_VERIFY_ON           ID_VERIFY_ON
#define EMS_COERCIVITY_DEFAULT  ID_COERCIVITY_DEFAULT
#define EMS_COERCIVITY_HICO     ID_COERCIVITY_HICO
#define EMS_COERCIVITY_LOCO     ID_COERCIVITY_LOCO
#define EMS_BITSPERCHAR_DEFAULT ID_BITSPERCHAR_DEFAULT
#define EMS_BITSPERCHAR_1       ID_BITSPERCHAR_1
#define EMS_BITSPERCHAR_5       ID_BITSPERCHAR_5
#define EMS_BITSPERCHAR_7       ID_BITSPERCHAR_7
#define EMS_BITSPERINCH_DEFAULT ID_BITSPERINCH_DEFAULT
#define EMS_BITSPERINCH_75      ID_BITSPERINCH_75
#define EMS_BITSPERINCH_210     ID_BITSPERINCH_210
#define EMS_PARITY_DEFAULT      ID_PARITY_DEFAULT
#define EMS_PARITY_OFF          ID_PARITY_OFF
#define EMS_PARITY_ODD          ID_PARITY_ODD
#define EMS_PARITY_EVEN         ID_PARITY_EVEN
#define EMS_LRC_DEFAULT         ID_LRC_DEFAULT
#define EMS_LRC_OFF             ID_LRC_OFF
#define EMS_LRC_ODD             ID_LRC_ODD
#define EMS_LRC_EVEN            ID_LRC_EVEN
LIBSPEC int WINAPI EncodeMagStripe(HANDLE, int, int, char *, int, int, int, int, int, int, int);
typedef int (WINAPI * PFNENCODEMAGSTRIPE)(HANDLE, int, int, char *, int, int, int, int, int, int, int);

LIBSPEC int WINAPI ReadMagStripe(HANDLE, PMSVDATA, int);
typedef int (WINAPI * PFNREADMAGSTRIPE)(HANDLE, PMSVDATA, int);

LIBSPEC int WINAPI DecodeMagData(PMSVDATA);
typedef int (WINAPI * PFNDECODEMAGDATA)(PMSVDATA);

enum {SEM_EJECT_ON, SEM_EJECT_OFF};
LIBSPEC int WINAPI SetEjectMode(HANDLE, int);
typedef int (WINAPI * PFNSETEJECTMODE)(HANDLE, int);

#define STATUS_READY   ID_STATUS_READY
#define STATUS_BUSY    ID_STATUS_BUSY
#define STATUS_ERROR   ID_STATUS_ERROR
#define STATUS_OFFLINE ID_STATUS_OFFLINE
LIBSPEC int WINAPI GetPrinterStatus(HANDLE);
typedef int (WINAPI * PFNGETPRINTERSTATUS)(HANDLE);

LIBSPEC int WINAPI GetEnduroInfo(HANDLE, PRINTER_INFO *);
LIBSPEC int WINAPI GetPrinterInfo(HANDLE, PRINTER_INFO *);
typedef int (WINAPI * PFNGETPRINTERINFO)(HANDLE, PPRINTER_INFO);
#define PFNGETENDUROINFO PFNGETPRINTERINFO

LIBSPEC int WINAPI PrintTestCard(HANDLE);
typedef int (WINAPI * PFNPRINTTESTCARD)(HANDLE);

LIBSPEC int WINAPI CleanPrinter(HANDLE);
typedef int (WINAPI * PFNCLEANPRINTER)(HANDLE);

LIBSPEC int WINAPI RestartPrinter(HANDLE);
typedef int (WINAPI * PFNRESTARTPRINTER)(HANDLE);

LIBSPEC int WINAPI FlipCard(HANDLE);
typedef int (WINAPI * PFNFLIPCARD)(HANDLE);

#define SMART_MODE_DEFAULT ID_SMART_DEFAULT
#define SMART_MODE_PLATEN  ID_SMART_PLATEN
LIBSPEC int WINAPI SetSmartMode(HANDLE, int);
typedef int (WINAPI * PFNSETSMARTMODE)(HANDLE, int);

LIBSPEC int WINAPI SetSmartLocation(HANDLE, int);
typedef int (WINAPI * PFNSETSMARTLOCN)(HANDLE, int);

LIBSPEC int WINAPI EraseCard(HANDLE, int, int, int, int);
typedef int (WINAPI * PFNERASECARD)(HANDLE, int, int, int, int);

#define ERASE_SPEED_THOROUGH ID_ERASE_THOROUGH
#define ERASE_SPEED_QUICK    ID_ERASE_QUICK
LIBSPEC int WINAPI SetEraseSpeed(HANDLE, int);
typedef int (WINAPI * PFNSETERASESPEED)(HANDLE, int);

#define API_VERSION SDKVERSION
LIBSPEC int WINAPI GetAPIVersion(HANDLE, API_VERSION *);
typedef int (WINAPI * PFNGETAPIVERSION)(HANDLE, API_VERSION*);

#define ERROR_CAN ID_RESPONSE_CAN
#define ERROR_OKY ID_RESPONSE_OK
LIBSPEC int WINAPI ErrorResponse(HANDLE, int);
typedef int (WINAPI * PFNERRORRESPONSE)(HANDLE, int);

#define PRINTER_RIOTANGO ID_RIOTANGO
#define PRINTER_AOTA     ID_AOTA
#define PRINTER_ENDURO   ID_ENDURO
LIBSPEC int WINAPI GetPrinterType(HANDLE);
typedef int (WINAPI * PFNGETPRINTERTYPE)(HANDLE);

#define PORT_UNKNOWN  ID_PORT_UNKNOWN
#define PORT_USB      ID_PORT_USB
#define PORT_RESERVED ID_PORT_RESERVED
#define PORT_TCPIP    ID_PORT_TCPIP
#define PORT_FILE     ID_PORT_FILE
LIBSPEC int WINAPI GetConnectionType(HANDLE);
typedef int (WINAPI * PFNCONNECTIONTYPE)(HANDLE);

//==================================
//== Helix Deprecated Definitions ==
//==================================

#define HELIX_STRING          ULTIMA_STRING
#define HELIX_INFO			  ULTIMA_INFO
#define PHELIX_INFO			  PULTIMA_INFO
#define ID_HELIX			  ID_ULTIMA
#define ID_MOD_HELIX		  ID_MOD_ULTIMA
#define HELIX_RAW             TRACK_DATA
#define ID_HELIXMAG           ID_MAGDATA2
#define PID_HELIXMAG		  PID_MAGDATA2
#define HELIX_NONE            ULTIMA_NONE           
#define HELIX_PERCEPTUAL      ULTIMA_PERCEPTUAL     
#define HELIX_SATURATION      ULTIMA_SATURATION     
#define HELIX_RELCOLORIMETRIC ULTIMA_RELCOLORIMETRIC
#define HELIX_ABSCOLORIMETRIC ULTIMA_ABSCOLORIMETRIC
#define HELIX_ICC_EXTERNAL    ULTIMA_ICC_EXTERNAL   

//##################################################################################################################################

#ifdef __cplusplus
}
#endif // __cplusplus

#endif // __MAGAPI_H__